<?php

namespace App\Controllers;

use App\Controllers\BaseController;
use App\Models\RestaurantModel;
use App\Models\UserModel;

class Auth extends BaseController
{
    public function login()
    {
        // load restaurants
        $restaurants_model = new RestaurantModel();
        $restaurants = $restaurants_model->select('id, name')->findAll();
        $data['restaurants'] = $restaurants;

        // validation errors
        $data['validation_errors'] = session()->getFlashdata('validation_errors');
        $data['select_restaurant'] = session()->getFlashdata('select_restaurant');

        // login error
        $data['login_error'] = session()->getFlashdata('login_error');

        return view('auth/login_frm', $data);
    }

    public function login_submit()
    {
        // form validation
        $validation = $this->validate([
            'text_email' => [
                'label' => 'E-mail',
                'rules' => 'required|min_length[3]|max_length[100]',
                'errors' => [
                    'required' => 'O campo {field} é obrigatório.',
                    'min_length' => 'O campo {field} deve ter, no mínimo, {param} caracteres.',
                    'max_length' => 'O campo {field} deve ter, no máximo, {param} caracteres.',
                ]
            ],
            'text_password' => [
                'label' => 'Senha',
                'rules' => 'required|min_length[6]|max_length[20]',
                'errors' => [
                    'required' => 'O campo {field} é obrigatório.',
                    'min_length' => 'O campo {field} deve ter, no mínimo, {param} caracteres.',
                    'max_length' => 'O campo {field} deve ter, no máximo, {param} caracteres.',
                ]
            ]
            
        ]);

        if(!$validation){
            return redirect()->back()->withInput()->with('validation_errors', $this->validator->getErrors());
        }

        // check login
        $email = $this->request->getPost('text_email');
        $password = $this->request->getPost('text_password');
        
        $user_model = new UserModel();
        $user = $user_model->check_for_login($email, $password);

        if(!$user){
            return redirect()->back()->withInput()->with('login_error', 'Usuário ou senha inválidos.');
        }

        // set session
        $user_data = [
            'id' => $user->id,
            'name' => $user->name,
            'username' => $user->username,
            'email' => $user->email,
            'phone' => $user->phone,
            'roles' => $user->roles,
            'id_restaurant' => $user->id_restaurant,
            'restaurant_name' => $user->restaurant_name ?? null, // optional restaurant name
        ];
     
        session()->set('user', $user_data);

        return redirect()->to('/');

    }

    public function logout()
    {
        session()->destroy();
        return redirect()->to('/');
    }

    // --------------------------------------------------------
    // finish registration
    // --------------------------------------------------------
    public function finish_registration($purl_code = null)
    {
        if(empty($purl_code)){
            return redirect()->to('auth/login');
        }

        // check if purl code exists
        $user_model = new UserModel();

        // get information from purl (including restaurant name)
        $user = $user_model->select('users.*, restaurants.name as restaurant_name')
                        ->where('users.code', $purl_code)
                        ->join('restaurants', 'restaurants.id = users.id_restaurant')
                        ->first();
        if(!$user){
            return redirect()->to('auth/login');
        }

        // place user data in session
        $new_user_data = [
            'id' => $user->id,
            'name' => $user->name,
            'username' => $user->username,
            'email' => $user->email,
            'phone' => $user->phone,
            'roles' => $user->roles,
            'id_restaurant' => $user->id_restaurant,
            'restaurant_name' => $user->restaurant_name,
        ];

        session()->set('new_user', $new_user_data);

        // redirect to the finish registration form (define password)
        return redirect()->to('/auth/define_password');
    }

    public function define_password()
    {
        // check if new user exist in session
        if(!session()->has('new_user')){
            return redirect()->to('auth/login');
        }

        // form validation
        $data['validation_errors'] = session()->getFlashdata('validation_errors');

        return view('auth/define_password_frm', $data);
    }

    public function define_password_submit()
    {
        // form validation
        $validation = $this->validate($this->_define_new_password_validation_rules());

        if (!$validation) {
            return redirect()->back()->withInput()->with('validation_errors', $this->validator->getErrors());
        }

        // update user password and active state
        $user_model = new UserModel();
        $id_user = session()->new_user['id'];
        $password = $this->request->getPost('text_password');
        $user_model->update($id_user, [
            'passwrd' => password_hash($password, PASSWORD_DEFAULT),
            'code' => null,
            'active' => 1,
            'updated_at' => date('Y-m-d H:i:s')
        ]);

        // remove new user from session
        session()->remove('new_user');

        // redirect to welcome page
        return redirect()->to('/auth/welcome')->with('success', true);
    }

    public function welcome()
    {
        // check if success is in session as flashdata
        if(!session()->getFlashdata('success')){
            return redirect()->to('/auth/login');
        }

        // display welcome page
        return view('auth/welcome');
    }

    
    // --------------------------------------------------------
    // user profile
    // --------------------------------------------------------
    public function profile()
    {
        // get user data
        $user_model = new UserModel();
        $user = $user_model->find(session()->user['id']);

        // check if user is valid
        if(!$user){
            return redirect()->to('/');
        }

        $user->role = json_decode($user->roles)[0];

        // form validation
        $data['validation_errors'] = session()->getFlashdata('validation_errors');

        // success profile message
        $data['profile_success'] = session()->getFlashdata('profile_success');
        
        // success new password message
        $data['password_success'] = session()->getFlashdata('password_success');

        // server error
        $data['server_error'] = session()->getFlashdata('server_error');

        // display profile page
        $data['title'] = 'Perfil do Usuário';
        $data['page'] = 'Perfil do Usuário';
        $data['user'] = $user;

        return view('auth/profile', $data);
    }

    public function profile_submit()
    {
        // form validation
        $validation = $this->validate($this->_profile_form_validation());

        if(!$validation){
            return redirect()->back()->withInput()->with('validation_errors', $this->validator->getErrors());
        }

        $user_model = new UserModel();

        // check if there is another user with the same email address
        $other_user = $user_model->where('id !=', session()->user['id'])
                                 ->where('email', $this->request->getPost('text_email'))
                                 ->get()->getResultArray();
        if($other_user){
            return redirect()->back()->withInput()->with('server_error', 'Já existe outro usuário com o mesmo email.');
        }

        // update user data
        $user_model->update(session()->user['id'], [
            'name' => $this->request->getPost('text_name'),
            'email' => $this->request->getPost('text_email'),
            'phone' => $this->request->getPost('text_phone'),
            'updated_at' => date('Y-m-d H:i:s'),
        ]);

        // update session
        $user = session()->user;
        $user['name'] = $this->request->getPost('text_name');
        $user['email'] = $this->request->getPost('text_email');
        $user['phone'] = $this->request->getPost('text_phone');
        session()->set('user', $user);

        // redirect to profile page
        return redirect()->to('/auth/profile')->with('profile_success', true);
    }

    public function change_password_submit()
    {
        // form validation
        $validation = $this->validate($this->_profile_change_password_form_validation());

        if(!$validation){
            return redirect()->back()->withInput()->with('validation_errors', $this->validator->getErrors());
        }

        // check if current password is correct
        $user_model = new UserModel();
        $user = $user_model->find(session()->user['id']);
        $password = $this->request->getPost('text_password');
        if(!password_verify($password, $user->passwrd)){
            return redirect()->back()->withInput()->with('validation_errors', ['text_password' => 'Senha atual está incorreta.']);
        }

        // update the user password
        $new_password = $this->request->getPost('text_new_password');
        $user_model->update(session()->user['id'], [
            'passwrd' => password_hash($new_password, PASSWORD_DEFAULT),
            'updated_at' => date('Y-m-d H:i:s')
        ]);

        // redirect to profile page
        return redirect()->to('/auth/profile')->with('password_success', true);
    }

    public function forgot_password()
    {
        // display forgot password page
        
        // load restaurants
        $restaurants_model = new RestaurantModel();
        $restaurants = $restaurants_model->select('id, name')->findAll();
        $data['restaurants'] = $restaurants;

        // form validation
        $data['validation_errors'] = session()->getFlashdata('validation_errors');

        return view('auth/forgot_password_frm', $data);
    }

    public function forgot_password_submit()
    {
        // form validation
        $validation = $this->validate($this->_forgot_password_validation_rules());

        if (!$validation) {
            return redirect()->back()->withInput()->with('validation_errors', $this->validator->getErrors());
        }

        // check if email exists
        $id_restaurant = Decrypt($this->request->getPost('select_restaurant'));
        $email = $this->request->getPost('text_email');
        $user_model = new UserModel();
        $user = $user_model->where('email', $email)
                           ->where('id_restaurant', $id_restaurant)
                           ->first();
        
        // always show success message to avoid give information about the existance of the email
        if(!$user){
            $this->_show_forgot_password_success_message();
            return;
        }

        // generate random hash code for purl
        $code = bin2hex(random_bytes(16));

        // update user code
        $user_model->update($user->id, [
            'code' => $code,
            'updated_at' => date('Y-m-d H:i:s')
        ]);

        $data = [
            'id' => $user->id,
            'name' => $user->name,
            'email' => $user->email,
            'code' => $code
        ];

        // send email to user to recover the password
        $this->_send_email($data);

        $this->_show_forgot_password_success_message();
    }

    private function _send_email($data)
    {
        // prepare purl
        $data['purl'] = site_url('/auth/redefine_password/' . $data['code']);

        // config email
        $email = \Config\Services::email();
        $email->setFrom('cigburger@cigburger.com', 'CigBurger');
        $email->setTo($data['email']);
        $email->setSubject('CigBurger - Recuperação de senha');
        $email->setMailType('html');
        $email->setMessage(view('emails/email_recover_password', $data));

        // send e-mail and return true or false
        return $email->send();
        
    }

    private function _show_forgot_password_success_message()
    {
        // display forgot password success page
        echo view('auth/forgot_password_success');
    }

    // --------------------------------------------------------
    // redefine password
    // --------------------------------------------------------
    public function redefine_password($purl_code = null)
    {
        // check if purl code is empty
        if(empty($purl_code)){
            return redirect()->to('/auth/login');
        }

        // check if purl code exists
        $user_model = new UserModel();
        $user = $user_model->where('code', $purl_code)->first();
        if(!$user){
            return redirect()->to('/auth/login');
        }

        // display redefine password page
        $data['purl_code'] = $purl_code;
        $data['validation_errors'] = session()->getFlashdata('validation_errors');

        return view('auth/redefine_password_frm', $data);
    }

    public function redefine_password_submit()
    {
        // form validation
        $validation = $this->validate($this->_define_reset_password_validation_rules());

        if (!$validation) {
            return redirect()->back()->withInput()->with('validation_errors', $this->validator->getErrors());
        }

        // update user password
        $user_model = new UserModel();
        $code = $this->request->getPost('purl_code');
        $password = $this->request->getPost('text_password');
        $user_model->where('code', $code)
                    ->set('passwrd', password_hash($password, PASSWORD_DEFAULT))
                    ->set('code', null)
                    ->set('updated_at', date('Y-m-d H:i:s'))
                    ->update();

        // display success page
        return view('auth/redefine_password_success');
    }

    // --------------------------------------------------------
    // form validation rules
    // --------------------------------------------------------
    private function _define_new_password_validation_rules()
    {
        return [
            'text_password' => [
                'label' => 'Senha',
                'rules' => 'required|min_length[8]|max_length[16]|regex_match[/(?=.*[a-z])(?=.*[A-Z])(?=.*\d)/]',
                'errors' => [
                    'required' => 'O campo {field} é obrigatório.',
                    'min_length' => 'O campo {field} deve ter no mínimo {param} caracteres.',
                    'max_length' => 'O campo {field} deve ter no máximo {param} caracteres.',
                    'regex_match' => 'O campo {field} deve ter pelo menos uma letra minúscula, uma maiúscula e um algarismo.'
                ]
            ],
            'text_password_confirm' => [
                'label' => 'Confirmar senha',
                'rules' => 'required|matches[text_password]',
                'errors' => [
                    'required' => 'O campo {field} é obrigatório.',
                    'matches' => 'O campo {field} deve ser igual ao campo Senha.'
                ]
            ],
        ];
    }

    private function _profile_form_validation()
    {
        return [
            'text_name' => [
                'label' => 'Nome do usuário',
                'rules' => 'required|min_length[3]|max_length[50]',
                'errors' => [
                    'required' => 'O campo {field} é obrigatório.',
                    'min_length' => 'O campo {field} deve ter no mínimo {param} caracteres.',
                    'max_length' => 'O campo {field} deve ter no máximo {param} caracteres.'
                ]
            ],
            'text_email' => [
                'label' => 'E-mail',
                'rules' => 'required|valid_email|min_length[5]|max_length[50]',
                'errors' => [
                    'required' => 'O campo {field} é obrigatório.',
                    'valid_email' => 'O campo {field} deve conter um e-mail válido.',
                    'min_length' => 'O campo {field} deve ter no mínimo {param} caracteres.',
                    'max_length' => 'O campo {field} deve ter no máximo {param} caracteres.'
                ]
            ],
            'text_phone' => [
                'label' => 'Telefone',
                'rules' => 'required|regex_match[/^[9]{1}\d{8}$/]',
                'errors' => [
                    'required' => 'O campo {field} é obrigatório.',
                    'regex_match' => 'O campo {field} deve conter um número de telefone válido.',
                ]
            ]
        ];
    }

    private function _profile_change_password_form_validation()
    {
        return [
            'text_password' => [
                'label' => 'Senha',
                'rules' => 'required|min_length[8]|max_length[16]|regex_match[/(?=.*[a-z])(?=.*[A-Z])(?=.*\d).*/]',
                'errors' => [
                    'required' => 'O campo {field} é obrigatório.',
                    'min_length' => 'O campo {field} deve ter no mínimo {param} caracteres.',
                    'max_length' => 'O campo {field} deve ter no máximo {param} caracteres.',
                    'regex_match' => 'O campo {field} deve conter pelo menos uma letra maiúscula, uma minúscula e um algarismo.'
                ]
            ],
            'text_new_password' => [
                'label' => 'Nova senha',
                'rules' => 'required|min_length[8]|max_length[16]|regex_match[/(?=.*[a-z])(?=.*[A-Z])(?=.*\d).*/]',
                'errors' => [
                    'required' => 'O campo {field} é obrigatório.',
                    'min_length' => 'O campo {field} deve ter no mínimo {param} caracteres.',
                    'max_length' => 'O campo {field} deve ter no máximo {param} caracteres.',
                    'regex_match' => 'O campo {field} deve conter pelo menos uma letra maiúscula, uma minúscula e um algarismo.'
                ]
            ],
            'text_new_password_confirm' => [
                'label' => 'Confirmar nova senha',
                'rules' => 'required|matches[text_new_password]',
                'errors' => [
                    'required' => 'O campo {field} é obrigatório.',
                    'matches' => 'O campo {field} deve ser igual ao campo Nova senha.'
                ]
            ]
        ];
    }

    private function _forgot_password_validation_rules()
    {
        return [
            'select_restaurant' => [
                'label' => 'Restaurante',
                'rules' => 'required',
                'errors' => [
                    'required' => 'O campo {field} é obrigatório.'
                ]
            ],
            'text_email' => [
                'label' => 'E-mail',
                'rules' => 'required|valid_email|min_length[5]|max_length[50]',
                'errors' => [
                    'required' => 'O campo {field} é obrigatório.',
                    'valid_email' => 'O campo {field} deve conter um e-mail válido.',
                    'min_length' => 'O campo {field} deve ter no mínimo {param} caracteres.',
                    'max_length' => 'O campo {field} deve ter no máximo {param} caracteres.'
                ]
            ]
        ];
    }

    private function _define_reset_password_validation_rules()
    {
        return [
            'purl_code' => [
                'label' => '',
                'rules' => 'required',
                'errors' => [
                    'required' => 'Aconteceu um erro na submissão do formulário'    // improvável a não ser de forma propositada
                ]
            ],
            'text_password' => [
                'label' => 'Senha',
                'rules' => 'required|min_length[8]|max_length[16]|regex_match[/(?=.*[a-z])(?=.*[A-Z])(?=.*\d).*/]',
                'errors' => [
                    'required' => 'O campo {field} é obrigatório.',
                    'min_length' => 'O campo {field} deve ter no mínimo {param} caracteres.',
                    'max_length' => 'O campo {field} deve ter no máximo {param} caracteres.',
                    'regex_match' => 'O campo {field} deve conter pelo menos uma letra maiúscula, uma minúscula e um algarismo.'
                ]
            ],
            'text_password_confirm' => [
                'label' => 'Confirmar Senha',
                'rules' => 'required|matches[text_password]',
                'errors' => [
                    'required' => 'O campo {field} é obrigatório.',
                    'matches' => 'O campo {field} deve ser igual ao campo Senha.'
                ]
            ]
        ];
    }
}
